/*  BCRAM_example.sas    (BrCa_Ram)                                         8/12/2011

    short example program comprising of 3 steps, which perform the following:

    Step 1 the creation and listing of a temporary sas input system file (ExampleIn)
           from a raw file (Sample.in) consisting of 26 hypothetical women's
           relative risk covariate information.

           input files are      BCPTCARE.in    or
                                QC_testing     slightly different formats. same results.

    Step 2 the involking of the sas macro  BrCa_Ram  to perform risk projection.

    Step 3 a simple post processing step.  The post processing is the listing of the
           temporary sas sytem file (Out_File) which contains the women's
           relative risk covariate information as well as projected absolute risk.


    Race codes:   1=WH   white 1983-87 SEER rates,  (rates used by NCI BrCa Risk Assessment Tool)
                  2=AA   african-american,
                  3=HI   hispanic,
                  4=NA   american indian/native alaskan
                  5=WO   white 1995-03 SEER rates   (rates for further research)

                  6=CH   chinese
                  7=JP   japanese
                  8=FI   filipino
                  9=HW   hawaiian
                 10=OP   othr pacific islander
                 11=OA   othr asian;
*/



*** Step 1 Input raw relative risk covariate file  "Sample.in"  and create a temporary
           sas system file with name of  "ExampleIn"  and list the file contents.
           The temporary sas file  "ExampleIn"  is used in step 2 for input to the sas
           macro  BrCa_Ram  which projects for absolute risk of BrCa;

options  nocenter  ls=115  ps=9999  notes;
options  FORMCHAR="|----|+|---+=|-/\<>*";

title1  'BCRAM_example.sas, sample sas program using   BrCa_Ram   for BrCa abs risk';

data    ExampleIn;             *** name of the sas system file which the macro parameter
                                  &In_File  should point to upon macro invocation;

        infile 'Sample.in'    firstobs=9;       *** "Sample.in"  is the RR covariate input file
                                                     firstobs=9  skips first 8 header
                                                     records on input file  "Sample.in";
            *** SAS variable names;

        input   IDD

                InitalAge
                ProjtnAge

                NBiop
                HP

                AgeMenarchy
                AgeFstLive
                Num_Rels

                Ethnicity;
run;


***      list the contents of the temporary sas sytem file  ExampleIn;

proc     print  data=ExampleIn;
id       IDD;
var      InitalAge  ProjtnAge  NBiop  HP  AgeMenarchy  AgeFstLive
         Num_Rels   Ethnicity;

title3  'Listing of the temporary input sas system file      "ExampleIn"';
title4  'Created from the raw relative risk covariate file   "Sample.in"';
run;


%include  "BrCa_RAM";                        *** copy  BrCa_RAM  Gail model SAS macro
                                                 here if you want the SAS macro in-line
                                                 with this program;

*** Step 2    Involking the sas macro  BrCa_RAM  to perform the BrCa projections.

              The temporary sas input  file is set to   "ExampleIn".
              The temporary sas output file is set to   "ExampleOut".

              The macro parameters  T1, T2, N_Biop, HyperPlasia, AgeMen,
              Age1st, N_Rels, and Race point to their corresponding sas variables
              on the sas file  "ExampleIn",  namely
              InitalAge, ProjtnAge, NBiop, HP, AgeMenarchy, AgeFstLive,
              Num_Rels and Ethnicity respectively.

              The macro parameter AbsRsk points to the sas variable Abs_Risk which
              will be added to the output sas file  "ExampleOut".  The output sas
              file will also contain all the variables on the input sas file.

              Macro          pointing      SAS file name or
              parameter      to            SAS variable name;

%BrCa_RAM    (In_File        =             ExampleIn    ,
              Out_File       =             ExampleOut   ,

              WID            =             IDD          ,

              T1             =             InitalAge    ,
              T2             =             ProjtnAge    ,

              N_Biop         =             NBiop        ,
              HyperPlasia    =             HP           ,

              AgeMen         =             AgeMenarchy  ,
              Age1st         =             AgeFstLive   ,
              N_Rels         =             Num_Rels     ,

              Race           =             Ethnicity    ,
              CharRace       =             CharRace     ,

              RR_Star1       =             RR_Star1     ,
              RR_Star2       =             RR_Star2     ,

              AbsRsk         =             Absolute_Risk,
              AbsRskAvg      =             AbsRisk_Avg);



*** Step 3    List the contents of the temporary output sas system file  "ExampleOut"
              which contains the projected absolute risk as well as the relative risk
              covariate values.  Note that any further processing requiring the
              projected absolute risk, must be performed on the output sas system file;

data  ExampleOut;                  *** output file from macro, defined by pointing the;
set   ExampleOut;                  *** macro parameter  &Out_File  to  "ExampleOut";

      file print;

      if (_N_ eq 1) then do;
         put "                                                           ";
         put "                        #  Hypr    HP  Age  Age    #       "
             "        RR        RR   AbsRisk   AbsRisk   ";
         put "     ID    T1    T2  Biop  plas    RR  Men  1st  Rel   Race"
             "    Age<50   Age>=50        (%)  AvgWm(%)";
         put " ";
      end;

      *** all variables below take on their SAS variable names, not their macro names;
      *** see SAS variable names defined in Step 1;

      if (_n_ le 100) then

         put IDD                  7.0
             InitalAge            6.1
             ProjtnAge            6.1

             NBiop                6.0
             HP                   6.0
             R_Hyp                6.2

             AgeMenarchy          5.0
             AgeFstLive           5.0
             Num_Rels             5.0

            "  "
             Ethnicity            2.0
            "="
             CharRace        $char2.

             RR_Star1            10.4
             RR_Star2            10.4

             Absolute_Risk       10.4
             AbsRisk_Avg         10.4;


title3   "Listing of the first 100 records in temporary output sas system file  ExampleOut";
title4   "Further analysis depending on the projected abs risk must be performed using the";
title5   "output sas system file which is invoked by the sas macro parameter  'Out_File'";
run;


proc    means  data=ExampleOut  n sum mean stderr min max median;
var     Absolute_Risk
        AbsRisk_Avg;

title3 " ";
run;
